/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.ComponentModel;
using System.Globalization;

using Borland.Eco.Subscription;
using Borland.Eco.UmlRt;
using Borland.Eco.Services;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Exceptions;
using Borland.Eco.Globalization;

namespace Borland.Eco.Handles
{
	public abstract class ElementHandle: System.ComponentModel.Component, IStaticContext, IEcoServiceProvider, ISubscribableElementProvider, IList, ITypedList, IBindingList
	{
#if CF
		protected bool DesignMode = false;
#endif
		private class VariablesChangedAdapter: SubscriberAdapterBase
		{
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null) throw new ArgumentNullException("actualSubscriber"); // do not localize
				((ElementHandle)actualSubscriber).StaticContextChanged();
			}

			public VariablesChangedAdapter(object subscriber): base(subscriber) {}
		}

		protected override void Dispose(bool disposing)
		{
			try
			{
				if (disposing)
				{
					Enabled = false;
					m_InternalElement = null;
					m_Variables = null;
				}
			}
			finally
			{
				base.Dispose(disposing);
			}
		}

		/// <summary>
		/// Gets the list used for databinding.
		/// </summary>
		public IList GetList()
		{
			return RenderedListDescriptor.List;
		}

		private HandleAdapter m_ListDescriptor;
		private HandleAdapter RenderedListDescriptor
		{
			get
			{
				if (m_ListDescriptor == null)
				{
					m_ListDescriptor = new HandleAdapter(this);
					RenderedListDescriptor.SetupProperties(Columns, AddDefaultProperties, AddDefaultNestings, AddExternalId, MemberVisibility, Nestings, DesignMode);
				}
				return m_ListDescriptor;
			}
		}

		private ColumnCollection m_Columns;

		/// <summary>
		/// Returns true if the column collection of the handle contains a columned name 'name'.
		/// </summary>
		private bool ColumnNameExists(string name)
		{
			for (int i = 0; i < Columns.Count; i++)
				if (string.Compare(name, Columns[i].Name, true, CultureInfo.InvariantCulture) == 0) // case insensitive
					return true;
			return false;
		}

		///<summary>
		/// Creates a new column based on the information of the <see cref="IStructuralFeature"/>.
		///</summary>
		private static AbstractColumn MakeNewColumn(IStructuralFeature sf)
		{
			Column column = new Column();
			column.Expression = "self." + sf.Name; // do not localize
			column.Name = sf.Name;
			column.Nested = (sf is IAssociationEnd);
			return column;
		}

		private static bool ShowMember(VisibilityKind visibility, MemberVisibility memberVisibility)
		{
			if (visibility == VisibilityKind.Private_ && !(memberVisibility == MemberVisibility.AllMembers)) return false;

			if (visibility == VisibilityKind.Protected_ &&
				!((memberVisibility == MemberVisibility.ProtectedOrHigher) ||
					(memberVisibility == MemberVisibility.AllMembers))) return false;
			return true;
		}

		///<summary>
		/// <para>Creates columns for all structural features with a name that does not
		/// match an existing column.</para>
		/// <para>AssociationEnds will be nested.</para>
		///
		/// <para>If the type is <see cref="IPrimitiveType"/> will get an unnamed column.</para>
		///
		/// <para>It is used by the designer/action verb.</para>
		///</summary>
		public void CreateDefaultColumns()
		{
			IClassifier classifier = HandleUtils.ItemType(((IStaticContext)this).StaticUmlType);

			IClass c = classifier as IClass;
			if (c != null)
			{
				for (int i = 0; i <= c.EcoClass.AllStructuralFeatures.Count - 1; i++)
				{
					IStructuralFeature sf = c.EcoClass.AllStructuralFeatures[i];
					if (!ShowMember(sf.Visibility, MemberVisibility)) continue;
					if (sf is IAttribute)
					{
						if (!ColumnNameExists(sf.Name))
							Columns.Add(MakeNewColumn(sf));
					}
					else if ((sf is IAssociationEnd))
					{
						IAssociationEnd role = (IAssociationEnd)sf;
						if(((role.Name).Length > 0) && role.IsNavigable && !ColumnNameExists(role.Name))
							Columns.Add(MakeNewColumn(sf));
					}
				}
			}
			else if (classifier is IPrimitiveType)
			{
				Column column = new Column();
				column.Expression = "self"; // do not localize
				Columns.Add(column);
			}
		}

		/// <summary>
		/// <para>This property affects the appearance of the list returned by <see cref="GetList"/>
		/// and used when binding .Net components.</para>
		/// <para>Each column in the collection will add a property to the list in
		/// addition to the ones defined by <see cref="AddDefaultProperties"/> and <see cref="AddDefaultNestings"/>.</para>
		/// </summary>
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyColumns")]
		public ColumnCollection Columns
		{
			get
			{
				if (m_Columns == null)
				{
					m_Columns = new ColumnCollection(new ItemStaticContext(this));
					m_Columns.OnChange += new EventHandler(ColumnsOrNestingsChanged);
				}
				return m_Columns;
			}
		}

		private NestingCollection m_Nestings;

		/// <summary>
		/// <para>This property affects the appearance of the list returned by <see cref="GetList"/>
		/// and used when binding .Net components.</para>
		///
		/// <para>Each nesting in the collection will add a list valued property to the list in
		/// addition to the ones defined by <see cref="AddDefaultProperties"/> and <see cref="AddDefaultNestings"/>.</para>
		///
		/// <para>Collection of definitions of the sub-properties for properties
		/// that are themselves lists. i.e. that have <see cref="AbstractColumn.Nested"/> = true.</para>
		/// </summary>
		[Editor("CollectionEditor", "UITypeEditor")]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyNestings")]
		public NestingCollection Nestings
		{
			get
			{
				if (m_Nestings == null)
				{
					m_Nestings = new NestingCollection(this);
					m_Nestings.OnChange += new EventHandler(ColumnsOrNestingsChanged);
				}
				return m_Nestings;
			}
		}

		private bool m_AddDefaultProperties = true;

		/// <summary>
		/// <para>This property affects the appearance of the list returned by <see cref="GetList"/>
		/// and used when binding .Net components.</para>
		/// <para>Set this property to true to add a property for each ECO attribute on
		/// the objects in the list.</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyAddDefaultProperties")]
		[DefaultValue(true)]
		public bool AddDefaultProperties
		{
			get { return m_AddDefaultProperties; }
			set
			{
				if (value != m_AddDefaultProperties)
				{
					m_AddDefaultProperties = value;
					ColumnsOrNestingsChanged(this, EventArgs.Empty);
				}
			}
		}

		private bool m_AddExternalId;

		/// <summary>
		/// <para>Adds a column with the name "ExternalId" of type string. This can be used to find
		/// the object again (via <see cref="IExternalIdService"/>).</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyAddExternalId")]
		[DefaultValue(false)]

		public bool AddExternalId
		{
			get { return m_AddExternalId; }
			set
			{
				if (value != m_AddExternalId)
				{
					m_AddExternalId = value;
					ColumnsOrNestingsChanged(this, EventArgs.Empty);
				}
			}
		}

		private bool m_AddDefaultNestings;

		/// <summary>
		/// <para>This property affects the appearance of the list returned by GetList
		/// and used when binding .Net components.</para>
		/// <para>Set this property to true to add a list-valued property for each ECO association end on
		/// the objects in the list.</para>
		/// <para>These are expandable by e.g. DataGrid.</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyAddDefaultNestings")]
		[DefaultValue(false)]
		public bool AddDefaultNestings
		{
			get { return m_AddDefaultNestings; }
			set
			{
				if (value != m_AddDefaultNestings)
				{
					m_AddDefaultNestings = value;
					ColumnsOrNestingsChanged(this, EventArgs.Empty);
				}
			}
		}

		private MemberVisibility m_MemberVisibility = MemberVisibility.PublicOnly;
		/// <summary>
		/// <para>This property affects which columns the handle exposes 
		/// based on the modeled visibility of the members.</para>
		/// <para>CreateDefaultColumns as well as DefaultColumns and DefaultNestings will act on this setting.</para>
		/// </summary>		
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyMemberVisibility")]
		[DefaultValue(MemberVisibility.PublicOnly)]
		public MemberVisibility MemberVisibility
		{
			get { return m_MemberVisibility; }
			set
			{
				if (value != m_MemberVisibility)
				{
					m_MemberVisibility = value;
					ColumnsOrNestingsChanged(this, EventArgs.Empty);
				}
			}
		}

		private void ColumnsOrNestingsChanged(System.Object dummy1, EventArgs dummy2)
		{
			if (m_ListDescriptor != null)
				RenderedListDescriptor.SetupProperties(Columns, AddDefaultProperties, AddDefaultNestings, AddExternalId, MemberVisibility, Nestings, DesignMode);
		}

		/// <summary>
		/// Event handling all the derivation events for all EventDerivcecolumns owned by the handle.
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyDeriveValue")]
		public event DeriveEventHandler DeriveValue
		{
			add { RenderedListDescriptor.TupleDescriptor.DeriveValue += value; }
			remove { RenderedListDescriptor.TupleDescriptor.DeriveValue -= value; }
		}

		/// <summary>
		/// Event handling all the reverse derivation events for all EventDerivcecolumns that are not ReadOnly owned by the handle.
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyDeriveValue")]
		public event ReverseDeriveEventHandler ReverseDeriveValue
		{
			add { RenderedListDescriptor.TupleDescriptor.ReverseDeriveValue += value; }
			remove { RenderedListDescriptor.TupleDescriptor.ReverseDeriveValue -= value; }
		}

		/// <summary>
		/// Creates an object for IElement of the same type as the objects in the bindinglist.
		/// </summary>
		public object RenderElement(IElement element)
		{
			return RenderedListDescriptor.GetRenderedTupleForElement(element);
		}

		/// <summary>
		/// <para>Given a row in the bindinglist or an object created from an element as returned by RenderElement
		/// this method sets the value of a named property (which is defined as column.Name except for auto-properties).</para>
		/// <para>Primarily intended for setting values in ASP apps.</para>
		/// </summary>
		public void SetRenderedElementProperty(object item, string propertyName, object value, bool autoConvert)
		{
			PropertyDescriptor descriptor = ((ITypedList)this).GetItemProperties(null)[propertyName];
#if !CF
			if (autoConvert)
				value = descriptor.Converter.ConvertFrom(value);
#endif
			descriptor.SetValue(item, value);
		}

		protected abstract IEcoTypeSystem GetTypeSystem();
		IEcoTypeSystem IStaticContext.TypeSystem
		{
			get { return GetTypeSystem(); }
		}

		protected abstract IOclTypeService GetOclTypeService();
		protected abstract IOclPsTypeService GetOclPsTypeService();
		protected abstract IActionLanguageTypeService GetActionLanguageTypeService();

		IOclTypeService IStaticContext.OclTypeService
		{
			get { return GetOclTypeService(); }
		}

		IOclPsTypeService IStaticContext.OclPsTypeService
		{
			get { return GetOclPsTypeService(); }
		}

		IActionLanguageTypeService IStaticContext.ActionLanguageTypeService
		{
			get { return GetActionLanguageTypeService(); }
		}

		IExternalVariableList IStaticContext.VariableList
		{
			get { return EffectiveVariables(); }
		}

		protected abstract IClassifier GetStaticUmlType();

		[Browsable(false)]
		public IClassifier StaticUmlType
		{
			get { return GetStaticUmlType(); }
		}

		object IEcoServiceProvider.GetEcoService(System.Type serviceType)
		{
			return GetEcoService(serviceType);
		}

		protected abstract object GetEcoService(System.Type serviceType);

		/// <summary>
		/// <para>Places subscription on StaticContext.</para>
		/// <para>Internal use</para>
		/// </summary>
		public void SubscribeToStaticContext(ISubscriber subscriber)
		{
			m_StaticContextChangedPublisher.AddSubscription(subscriber);
		}

		protected virtual void EnsureInternalElement()
		{
			// Do nothing
		}

		/// <summary>
		/// The ECO value of the handle.
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public IElement Element
		{
			get
			{
				if (Active)
				{
					EnsureInternalElement();
					return InternalElement;
				}
				else
					return null;
			}
		}

		/// <summary>
		/// <para>Places subscription on Element.</para>
		/// <para>Internal use</para>
		/// </summary>
		public void SubscribeToElement(ISubscriber subscriber)
		{
			m_ElementChangedPublisher.AddSubscription(subscriber);
		}

		/// <summary>
		/// Equivalent of calling EnsureBindingList on all handles.
		/// </summary>
		public static void EnsureAllBindingLists()
		{
			DisplayQueue.DisplayAll();
		}
		/// <summary>
		/// <para>Will make the BindingList current relative to its value.</para>
		/// <para>In WinForms apps this method is called automatically OnIdle.</para>
		/// <para>In ASP apps it needs to be called manually when required.</para>
		/// </summary>
		public void EnsureBindingList()
		{
			RenderedListDescriptor.EnsureCurrent();
		}

		private readonly Publisher m_ElementChangedPublisher = new Publisher();
		protected void ElementChanged()
		{
			m_ElementChangedPublisher.Send(this, EventArgs.Empty);
		}

		protected virtual IExternalVariableList EffectiveVariables()
		{
			if (Variables == null)
				return null;
			else
				return Variables.OclVariableCollection;
		}

		private readonly Publisher m_StaticContextChangedPublisher = new Publisher();
		protected virtual void StaticContextChanged()
		{
			m_StaticContextChangedPublisher.Send(this, EventArgs.Empty);
			if (m_ListDescriptor != null)
				RenderedListDescriptor.SetupProperties(Columns, AddDefaultProperties, AddDefaultNestings, AddExternalId, MemberVisibility, Nestings, DesignMode);
			ElementChanged();
		}

		protected void CheckType()
		{
			if ((Element != null) && (GetStaticUmlType() != null) && ! Element.UmlType.EcoClassifier.IsA(GetStaticUmlType()))
			{
				throw new EcoException(HandlesStringRes.sTypeMismatchInHandle(Element.UmlType.Name, GetStaticUmlType().Name));
			}
		}

		private	IElement m_InternalElement;
		protected IElement InternalElement
		{
			get { return m_InternalElement; }
			set
			{
				if ((value != null) && (GetStaticUmlType() != null) && ! value.UmlType.EcoClassifier.IsA(GetStaticUmlType()))
				{
					throw new EcoException(HandlesStringRes.sStaticTypeNotMatching(value.UmlType.Name, GetStaticUmlType().Name));
				}
				m_InternalElement = value;
			}
		}

		protected virtual bool RefersToComponent(object component)
		{
			return false;
		}

		/// <summary>
		/// A handle is active when <see cref="Enabled"/> is True, and the component to
		/// which the handle is connected (the EcoSpace for a root handle,
		/// another ElementHandle for a rooted handle) has its Active property
		/// set to True.
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public abstract bool Active
		{
			get;
		}

		private readonly Publisher m_ActiveChangedPublisher = new Publisher();
		protected virtual void ActiveChanged()
		{
			m_ActiveChangedPublisher.Send(this, EventArgs.Empty);
		}

		/// <summary>
		/// <para>Places subscription on Active.</para>
		/// <para>Mainly for internal use</para>
		/// </summary>
		public void SubscribeToActive(ISubscriber subscriber)
		{
			m_ActiveChangedPublisher.AddSubscription(subscriber);
		}

		private bool m_Enabled = true;
		/// <summary>
		/// <para>If set to False, the element is set to null and the Active property is
		/// set to False, effectively enabling the handle.</para>
		/// <para>However, the implementation of <see cref="IStaticContext"/> is still valid.</para>
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryPresentation")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyEnabled")]
		[DefaultValue(true)]
		public bool Enabled
		{
			get { return m_Enabled; }
			set
			{
				if (Enabled != value)
				{
					m_Enabled = value;
					ActiveChanged();
				}
			}
		}

		private VariablesChangedAdapter m_VariablesChangedAdapter;

		private OclVariables m_Variables;
		/// <summary>
		/// Set this property to refer to an OclVariables component defining the
		/// variables to be used by the handle. The variables are passed on to
		/// any connected handles by way of the VariableList property on IStaticContext.
		///
		/// Variables defined here will be available for expressions when defining columns and nestings as well as
		/// in the expression of the handle.
		/// </summary>
		///<exception cref="EcoException">Thrown if creating a circular reference when setting the new value.</exception>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryOCL")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyVariables")]
		[Browsable(true)]
		[DefaultValue(null)]
		public OclVariables Variables
		{
			get { return m_Variables; }
			set
			{
				if (value != Variables)
				{
					if ((value != null) && value.LinksToHandle(this))
						throw new EcoException(HandlesStringRes.sCircularReference); //(sCircularReference, [classname, 'SetVariables', 'name', 'value.name']); /// XXX component name
					m_Variables = value;
					StaticContextChanged();
					// Drop all subscriptions by dropping old adapter
					if (m_VariablesChangedAdapter != null)
						m_VariablesChangedAdapter.Deactivate();
					m_VariablesChangedAdapter = new VariablesChangedAdapter(this);
					if (value != null)
						((IExternalVariableList)value.OclVariableCollection).Subscribe(m_VariablesChangedAdapter);
				}
			}
		}

		#region IEnumerable Implementation

		IEnumerator IEnumerable.GetEnumerator()
		{
			return GetList().GetEnumerator();
		}

		#endregion

		#region ICollection Implementation

		bool ICollection.IsSynchronized
		{
			get { return GetList().IsSynchronized; }
		}

		int ICollection.Count
		{
			get { return GetList().Count; }
		}

		void ICollection.CopyTo(Array array, int index)
		{
			GetList().CopyTo(array, index);
		}

		object ICollection.SyncRoot
		{
			get { return GetList().SyncRoot; }
		}

		#endregion

		#region IList Implementation

		bool IList.IsReadOnly
		{
			get { return GetList().IsReadOnly; }
		}

		object IList.this[int index]
		{
			get { return GetList()[index]; }
			set { GetList()[index] = value; }
		}

		void IList.RemoveAt(int index)
		{
			GetList().RemoveAt(index);
		}

		void IList.Insert(int index, object value)
		{
			GetList().Insert(index, value);
		}

		void IList.Remove(object value)
		{
			GetList().Remove(value);
		}

		bool IList.Contains(object value)
		{
			return GetList().Contains(value);
		}

		void IList.Clear()
		{
			GetList().Clear();
		}

		int IList.IndexOf(object value)
		{
			return GetList().IndexOf(value);
		}

		int IList.Add(object value)
		{
			return GetList().Add(value);
		}

		bool IList.IsFixedSize
		{
			get { return GetList().IsFixedSize; }
		}
		#endregion


		#region ITypedList Implementation

		private ITypedList GetTypedList()
		{
			return GetList() as ITypedList;
		}

		PropertyDescriptorCollection ITypedList.GetItemProperties(PropertyDescriptor[] listAccessors)
		{
			// TODO: Add ElementHandle.GetItemProperties implementation
			return GetTypedList().GetItemProperties(listAccessors);
		}

		string ITypedList.GetListName(PropertyDescriptor[] listAccessors)
		{
			return GetTypedList().GetListName(listAccessors);
		}

		#endregion

		#region IBindingList implementation

		private IBindingList GetBindingList()
		{
			return GetList() as IBindingList;
		}
		bool IBindingList.AllowEdit
		{
			get { return GetBindingList().AllowEdit; }
		}

		bool IBindingList.AllowNew
		{
			get { return GetBindingList().AllowNew; }
		}

		bool IBindingList.AllowRemove
		{
			get { return GetBindingList().AllowRemove; }
		}

		bool IBindingList.IsSorted
		{
			get { return GetBindingList().IsSorted; }
		}

		ListSortDirection IBindingList.SortDirection
		{
			get { return GetBindingList().SortDirection; }
		}

		PropertyDescriptor IBindingList.SortProperty
		{
			get { return GetBindingList().SortProperty; }
		}

		bool IBindingList.SupportsChangeNotification
		{
			get { return GetBindingList().SupportsChangeNotification; }
		}

		bool IBindingList.SupportsSearching
		{
			get { return GetBindingList().SupportsSearching; }
		}

		bool IBindingList.SupportsSorting
		{
			get { return GetBindingList().SupportsSorting; }
		}

		void IBindingList.AddIndex(PropertyDescriptor property)
		{
			GetBindingList().AddIndex(property);
		}

		object IBindingList.AddNew()
		{
			return GetBindingList().AddNew();
		}

		void IBindingList.ApplySort(PropertyDescriptor property, ListSortDirection direction)
		{
			GetBindingList().ApplySort(property, direction);
		}

		int IBindingList.Find(PropertyDescriptor property, object key)
		{
			return GetBindingList().Find(property, key);
		}

		void IBindingList.RemoveIndex(PropertyDescriptor property)
		{
			GetBindingList().RemoveIndex(property);
		}

		void IBindingList.RemoveSort()
		{
			GetBindingList().RemoveSort();
		}

		event ListChangedEventHandler IBindingList.ListChanged
		{
			add { GetBindingList().ListChanged += value; }
			remove { GetBindingList().ListChanged -= value; }
		}
		#endregion
	}
}
